include <BOSL2/std.scad>
include <BOSL2/constants.scad>
use <BOSL2/shapes3d.scad>
use <JetBrainsMono-Italic-VariableFont_wght.ttf>
use <NotoSansJP-Regular.otf>
use <PTMono-Regular.ttf>
use <NotoSansMono-Bold.ttf>

// Parameters
lens_focal_length = 50; //mm 75
lens_diameter = 50; //mm 30
lens_holder_edge_thickness = 2; //mm
lens_thickness = 2; //mm
lens_holder_thickness = 3; //mm
lens_holder_outwards_radius = 5; //mm

camera_inner_dimensions = [100,100,50]; //mm 

external_width = 2.5; //mm

image_plane_width = 0.8; // mm

// This equation can come in handy OA*f/(OA+f)=OA'
object_distances = [100,102.632,105.556,108.824,112.5,116.667,121.429,126.923,133.333,140.909,150,161.111,175,192.857,216.667,235.185,258.333,288.095,327.778,383.333,466.667,605.556,883.333,1716.67,"∞"]; // mm [300,325,350,375,400,450,500,600,750,1000,1500,3000,"∞"]

text_top_padding = 1; // mm
text_offset_coeff = -3.7; // -1.9
index_offset = true; // false
text_size = 3.3; // mm 2.6
text_angle = 180; // deg
text_font = "JetBrainsMono:style=Bold Italic"; //"JetBrainsMono:style=Bold";

jp_font = "NotoSansJP:style=Bold";
camera_name = "立方体";
spec_text_size = 5;
spec_text_padding = 5; // 15

attaching_mechanism_dimensions = [15,40,1.4]; // mm
attaching_mechanism_tolerance = 0.25; // mm (applied to both parts)

cyl_resolution = 200;
txt_resolution = 80;
crn_resolution = 60;

corner_rounding = 2;

// System
// Camera box
camera_box_dimensions = [camera_inner_dimensions.x+external_width*2,camera_inner_dimensions.y+external_width*2,camera_inner_dimensions.z+external_width];

// Lens holder
lens_holder_cyl_height = lens_thickness+lens_holder_thickness*2;
lens_holder_external_cyl_height = (lens_holder_cyl_height-external_width)/2;

// Image plane text
side_image_plane_text_count = ceil(len(object_distances)/2);

// Modules

function image_plane_calc(object_distance,focal_length)= (object_distance == "∞")? focal_length : -object_distance*focal_length/(focal_length-object_distance);


module image_plane_transform(object_distance,focal_length) { // mm
    if(object_distance == "∞"){
    translate([focal_length,0,0]) children();
        } else{
   OA_prime = -object_distance*focal_length/(focal_length-object_distance);
    translate([OA_prime,0,0]) children();
    }
}

function image_plane_text_dist_from_top (object_distance,focal_length,index) = -text_top_padding+(index_offset ? text_offset_coeff*(side_image_plane_text_count-index-1) : text_offset_coeff*image_plane_calc(object_distance,focal_length)-text_offset_coeff*focal_length);
    

module image_plane_text_transform(object_distance,focal_length,index){
    fixed_index = index%side_image_plane_text_count;
    has_looped = fixed_index<index;
    dist_from_top = image_plane_text_dist_from_top(object_distance,focal_length,fixed_index);
    translate([0, has_looped ? camera_inner_dimensions.y+external_width*2: 0, camera_inner_dimensions.z+external_width/2-(has_looped ? text_size : 0)])
    translate([external_width/2, (has_looped ? 0 : 1) * external_width/4, external_width/2])
    translate([0,0,dist_from_top])
    image_plane_transform(object_distance,focal_length)  children();
}

module image_plane_text(object_distance,focal_length,index){
    fixed_index = index%side_image_plane_text_count;
    has_looped = fixed_index<index;
    image_plane_text_transform(object_distance,focal_length,index) rotate([90,text_angle+(has_looped?180:0),(has_looped?180:0)])
    translate([0,0,(has_looped ? -1 : 0) * external_width/4])
    linear_extrude(external_width/4)
    if(object_distance != "∞"){
     text(format("{:-4d} mm; Υ: {:0.3f}", [object_distance, image_plane_calc(object_distance,focal_length)/-object_distance]),text_size, font = text_font, $fn=txt_resolution);
    } else {
     text(format("{:-4s} mm", [object_distance]),text_size, font = text_font, $fn=txt_resolution);
    }
    }

module image_plane(object_distance,focal_length,index) { // mm
    fixed_index = index%side_image_plane_text_count;
    has_looped = fixed_index<index;
    translate([external_width/2, external_width/2+external_width/4, external_width/2+external_width/4]) 
    image_plane_transform(object_distance,focal_length)
    cube([image_plane_width,camera_inner_dimensions.y+external_width/2,camera_inner_dimensions.z+external_width/2]);
    // Legend text
    image_plane_text(object_distance,focal_length,index);
    translate([image_plane_width/4, -external_width/4,0]) 
    image_plane_text_transform(object_distance,focal_length,index)
   translate([0,0,(has_looped ? text_size : 0)]) cube([image_plane_width/2,external_width/4,camera_inner_dimensions.z]);
}

module rounded_cuboid(dimensions,edges,rounding){
    cuboid(dimensions,anchor=[-1,-1,-1],rounding=rounding,edges=edges,$fn=crn_resolution);
}

module lens_transform(){
translate([0, external_width+camera_inner_dimensions.y/2, external_width+camera_inner_dimensions.z]) rotate([-90,0,-90])
children();}

module lens_cut_out(){
translate([-lens_holder_external_cyl_height, 0,0])
lens_transform() 
    cylinder(external_width/2-lens_thickness/2+lens_holder_external_cyl_height,lens_diameter/2-lens_holder_edge_thickness,lens_diameter/2, $fn=cyl_resolution);
    translate([external_width/2+lens_thickness/2, 0,0]) lens_transform()
    cylinder(external_width/2-lens_thickness/2+lens_holder_external_cyl_height,lens_diameter/2,lens_diameter/2-lens_holder_edge_thickness, $fn=cyl_resolution);
    translate([external_width/2-lens_thickness/2, 0,0]) lens_transform()
    cylinder(lens_thickness,lens_diameter/2,lens_diameter/2, $fn=cyl_resolution);
}

module half_cylinder(h,r1,r2){
difference(){cylinder(h,r1,r2, $fn=cyl_resolution);
translate([-r1-r2,(-r1-r2)*2,-0.5])
cube([(r1+r2)*2,(r1+r2)*2,h+1]);
}
}

difference(){
    rounded_cuboid(camera_box_dimensions,[BOTTOM+FRONT,BOTTOM+RIGHT,BOTTOM+LEFT,BOTTOM+BACK,FRONT+LEFT,FRONT+RIGHT,BACK+LEFT,BACK+RIGHT],corner_rounding);
    // Internal cavity
translate([external_width, external_width, external_width]) cube(camera_inner_dimensions);
    // Lens
    lens_cut_out();
    // Image planes
for(i = [0:len(object_distances)-1]){
    distance = object_distances[i];
    image_plane(distance,lens_focal_length,i);
    }
    // Camera specs
    // f-stop
    translate([spec_text_padding,external_width/4,spec_text_padding]) rotate([90,0,0])
    linear_extrude(external_width/4)
    text(format("f/   {:.2f}", [lens_focal_length/lens_diameter]),spec_text_size, font = text_font, $fn=txt_resolution);
    // focal length
    translate([spec_text_padding,external_width/4,spec_text_padding+spec_text_size*1.5]) rotate([90,0,0])
    linear_extrude(external_width/4)
    text(format("f: {:3d} mm", [lens_focal_length]),spec_text_size, font = text_font, $fn=txt_resolution);
    // Camera name
    translate([spec_text_padding,external_width/4,spec_text_padding+spec_text_size*1.5*2]) rotate([90,0,0])
    linear_extrude(external_width/4)
    text(camera_name, spec_text_size, font = jp_font, $fn=txt_resolution);
    // Attaching mechanism internal
    translate([camera_inner_dimensions.x+external_width-attaching_mechanism_tolerance, external_width+camera_inner_dimensions.y/2-(attaching_mechanism_dimensions.x+attaching_mechanism_tolerance-attaching_mechanism_tolerance/4), external_width+camera_inner_dimensions.z-(attaching_mechanism_dimensions.y+attaching_mechanism_tolerance)]) rounded_cuboid([attaching_mechanism_dimensions.z+attaching_mechanism_tolerance,(attaching_mechanism_dimensions.x+attaching_mechanism_tolerance),(attaching_mechanism_dimensions.y+attaching_mechanism_tolerance)],[FRONT+LEFT,FRONT+RIGHT,FRONT+BOTTOM,BACK+LEFT,BACK+RIGHT,BACK+BOTTOM,BOTTOM+LEFT,BOTTOM+RIGHT],(attaching_mechanism_dimensions.z+attaching_mechanism_tolerance)/4);
}

// Attaching mechanism external part
translate([camera_inner_dimensions.x+external_width, external_width+camera_inner_dimensions.y/2+attaching_mechanism_tolerance/4, external_width+camera_inner_dimensions.z]) rounded_cuboid([attaching_mechanism_dimensions.z-attaching_mechanism_tolerance,attaching_mechanism_dimensions.x-attaching_mechanism_tolerance,attaching_mechanism_dimensions.y-attaching_mechanism_tolerance],[FRONT+LEFT,FRONT+RIGHT,FRONT+TOP,BACK+LEFT,BACK+RIGHT,BACK+TOP,TOP+LEFT,TOP+RIGHT],(attaching_mechanism_dimensions.z-attaching_mechanism_tolerance)/4);

// Lens holder
difference(){
union(){
 translate([-lens_holder_external_cyl_height,0,0]) lens_transform() half_cylinder(lens_holder_external_cyl_height,lens_diameter/2,lens_diameter/2+lens_holder_outwards_radius);
 translate([external_width,0,0]) lens_transform() half_cylinder(lens_holder_external_cyl_height,lens_diameter/2+lens_holder_outwards_radius,lens_diameter/2);
 }
 lens_cut_out();
}