package piece

import (
	"errors"
	"fmt"
	"generate/config"
	"generate/utils"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"slices"
	"time"
)

type Piece struct {
	ID               string
	URL              string
	DateCreated      time.Time
	PositionAbsolute bool
	RelativeID       string
	InfoCardTitle    string
	InfoCardPosition int
	Position         utils.Coords
	AddWidth         bool
	AddHeight        bool
	Size             int
	Resolution       utils.Coords
	Video            bool
}

const (
	Half = iota
	Full
)

const (
	BRightSTop = iota
	BRightSBottom
	BLeftSTop
	BLeftSBottom
	BTopSLeft
	BTopSRight
	BBottomSLeft
	BBottomSRight
)

func ComputeDimensions(piece Piece) utils.Coords {
	height := 0.0

	if piece.Size == Full {
		height = config.GalleryHeight - config.GalleryPadding*2
	} else {
		height = (config.GalleryHeight-config.VerticalPadding)/2 - config.GalleryPadding
	}
	if slices.Contains([]int{BTopSLeft, BTopSRight, BBottomSLeft, BBottomSRight}, piece.InfoCardPosition) {
		height -= config.VerticalPadding + config.InfoCardHeight
	}
	height -= config.PiecePadding * 2
	width := height * piece.Resolution.X / piece.Resolution.Y
	return utils.Coords{X: width + config.PiecePadding*2, Y: height + config.PiecePadding*2}
}

func ComputePosition(offset utils.Coords, piece Piece, pieces []Piece) utils.Coords {
	offset.X += piece.Position.X
	offset.Y += piece.Position.Y
	if piece.PositionAbsolute {
		if slices.Contains([]int{BLeftSTop, BLeftSBottom}, piece.InfoCardPosition) {
			offset.X += config.InfoCardWidth + config.HorizontalPadding
		}
		return utils.Coords{X: offset.X, Y: offset.Y}
	} else {
		foundPiece := Piece{}
		isPieceFound := false
		for _, iPiece := range pieces {
			if iPiece.ID == piece.RelativeID {
				isPieceFound = true
				foundPiece = iPiece
			}
		}
		if !isPieceFound {
			log.Fatal("Invalid reference to piece ID: ", piece.RelativeID)
		}
		if piece.AddWidth {
			offset.X += ComputeDimensions(foundPiece).X + config.HorizontalPadding
			if slices.Contains([]int{BLeftSTop, BLeftSBottom, BRightSTop, BRightSBottom}, foundPiece.InfoCardPosition) {
				offset.X += config.InfoCardWidth + config.HorizontalPadding
			}
		}
		if piece.AddHeight {
			offset.Y += ComputeDimensions(foundPiece).Y + config.VerticalPadding
			if slices.Contains([]int{BTopSLeft, BTopSRight, BBottomSLeft, BBottomSRight}, foundPiece.InfoCardPosition) {
				offset.Y += config.InfoCardHeight + config.VerticalPadding
			}
		}
		return ComputePosition(offset, foundPiece, pieces)
	}
}

func GeneratePieceHTML(piece Piece, pieces []Piece) (string, float64) {
	pieceDimensions := ComputeDimensions(piece)
	piecePosition := ComputePosition(utils.Coords{}, piece, pieces)

	maxX := pieceDimensions.X + piecePosition.X
	if slices.Contains([]int{BRightSTop, BRightSBottom}, piece.InfoCardPosition) {
		maxX += config.HorizontalPadding + config.InfoCardWidth
	}
	piecePositionX := piecePosition.X
	piecePositionY := piecePosition.Y + config.GalleryPadding
	infoCardPositionX := piecePositionX
	infoCardPositionY := piecePositionY
	if slices.Contains([]int{BTopSLeft, BTopSRight}, piece.InfoCardPosition) {
		piecePositionY += config.VerticalPadding + config.InfoCardHeight
	}
	switch piece.InfoCardPosition {
	case BRightSTop:
		infoCardPositionX += pieceDimensions.X + config.HorizontalPadding
	case BRightSBottom:
		infoCardPositionX += pieceDimensions.X + config.HorizontalPadding
		infoCardPositionY += pieceDimensions.Y - config.InfoCardHeight
	case BLeftSTop:
		infoCardPositionX -= config.InfoCardWidth + config.HorizontalPadding
	case BLeftSBottom:
		infoCardPositionX -= config.InfoCardWidth + config.HorizontalPadding
		infoCardPositionY += pieceDimensions.Y - config.InfoCardHeight
	case BTopSLeft:
		//
	case BTopSRight:
		infoCardPositionX += pieceDimensions.X - config.InfoCardWidth
	case BBottomSLeft:
		infoCardPositionY += pieceDimensions.Y + config.VerticalPadding
	case BBottomSRight:
		infoCardPositionX += pieceDimensions.X - config.InfoCardWidth
		infoCardPositionY += pieceDimensions.Y + config.VerticalPadding
	}
	contentHTML := ""

	abs_input_path, err := filepath.Abs("../" + config.AssetsRootURL + piece.URL)
	if err != nil {
		log.Fatal(err)
	}
	abs_output_path, err := filepath.Abs("../" + config.AssetsCompressedRootURL + piece.ID)
	if err != nil {
		log.Fatal(err)
	}
	assetsImageCompressionArgs := make([]string, len(config.AssetsImageCompressionArgs)+4)
	for i := 0; i < len(config.AssetsImageCompressionArgs); i++ {
		assetsImageCompressionArgs[i+3] = config.AssetsImageCompressionArgs[i]
	}
	assetsImageCompressionArgs[0] = "-y"
	assetsImageCompressionArgs[1] = "-i"
	assetsImageCompressionArgs[2] = abs_input_path
	assetsImageCompressionArgs[len(assetsImageCompressionArgs)-1] = abs_output_path + ".webp"

	assetsVideoCompressionArgs := make([]string, len(config.AssetsVideoCompressionArgs)+4)
	for i := 0; i < len(config.AssetsVideoCompressionArgs); i++ {
		assetsVideoCompressionArgs[i+3] = config.AssetsVideoCompressionArgs[i]
	}
	assetsVideoCompressionArgs[0] = "-y"
	assetsVideoCompressionArgs[1] = "-i"
	assetsVideoCompressionArgs[2] = abs_input_path
	assetsVideoCompressionArgs[len(assetsVideoCompressionArgs)-1] = abs_output_path + ".webm"

	cmd := exec.Command("ffmpeg", assetsImageCompressionArgs...)
	if piece.Video {
		cmd = exec.Command("ffmpeg", assetsVideoCompressionArgs...)
	}
	if errors.Is(cmd.Err, exec.ErrDot) {
		cmd.Err = nil
	}
	if config.RecomputeCompression {
		cmd.Stdout, cmd.Stderr = os.Stdout, os.Stderr
		err = cmd.Run()
		if err != nil {
			fmt.Println(cmd.String())
			log.Fatal(err)
		}
	} else {
		fmt.Println(cmd.String())
	}
	if piece.Video {
		contentHTML = fmt.Sprintf(`<video width="%f" height="%f" autoplay muted playsinline loop><source src="%s" type="video/mp4" /></video>`, pieceDimensions.X-config.PiecePadding*2, pieceDimensions.Y-config.PiecePadding*2, config.AssetsCompressedRootURL+piece.ID+".webm")
	} else {
		contentHTML = fmt.Sprintf(`<img src="%s" style="width:%fpx;height:%fpx;"></img>`, config.AssetsCompressedRootURL+piece.ID+".webp", pieceDimensions.X-config.PiecePadding*2, pieceDimensions.Y-config.PiecePadding*2)
	}
	return fmt.Sprintf(`
	<div class="piece-container" style="left:%fpx;top:%fpx;width:%fpx;height:%fpx;">%s</div>
	<div class="info-card" style="left:%fpx;top:%fpx;"><div class="info-card-title">%s</div><div class="info-card-date">%s</div></div>
	`, piecePositionX, piecePositionY, pieceDimensions.X-config.PiecePadding*2, pieceDimensions.Y-config.PiecePadding*2, contentHTML, infoCardPositionX, infoCardPositionY, piece.InfoCardTitle, piece.DateCreated.Format("Mon 02 Jan 2006 15:04 MST")), maxX
}

var GalleryPieces = []Piece{

	{
		ID:               "jpkr30",
		URL:              "jpandkorea2025/P1239080.jpg",
		DateCreated:      time.Date(2025, 8, 12, 15, 35, 29, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: true,
		RelativeID:       "",
		InfoCardTitle:    "봉은사 3",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 1000, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr29",
		URL:              "jpandkorea2025/P1238955.jpg",
		DateCreated:      time.Date(2025, 8, 12, 15, 29, 20, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr30",
		InfoCardTitle:    "봉은사 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr28",
		URL:              "jpandkorea2025/P1238716.jpg",
		DateCreated:      time.Date(2025, 8, 12, 15, 23, 46, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr29",
		InfoCardTitle:    "봉은사 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr27",
		URL:              "jpandkorea2025/P1228444.jpg",
		DateCreated:      time.Date(2025, 8, 11, 19, 21, 14, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr28",
		InfoCardTitle:    "서울의 일몰",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr26",
		URL:              "jpandkorea2025/P1228260.jpg",
		DateCreated:      time.Date(2025, 8, 11, 18, 51, 53, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr27",
		InfoCardTitle:    "N서울타워",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr25",
		URL:              "jpandkorea2025/P1228193.jpg",
		DateCreated:      time.Date(2025, 8, 11, 18, 33, 59, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr26",
		InfoCardTitle:    "서울의 맑고 푸른 하늘",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr24",
		URL:              "jpandkorea2025/P1228117.jpg",
		DateCreated:      time.Date(2025, 8, 10, 16, 39, 04, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr25",
		InfoCardTitle:    "海の中道海浜公園",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr23",
		URL:              "jpandkorea2025/P1228022.jpg",
		DateCreated:      time.Date(2025, 8, 9, 16, 29, 13, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr24",
		InfoCardTitle:    "黒川温泉",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr22",
		URL:              "jpandkorea2025/P1227878.jpg",
		DateCreated:      time.Date(2025, 8, 8, 19, 47, 45, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr23",
		InfoCardTitle:    "NHK Nagasaki 5",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr21",
		URL:              "jpandkorea2025/P1217663.jpg",
		DateCreated:      time.Date(2025, 8, 8, 19, 34, 42, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr22",
		InfoCardTitle:    "NHK Nagasaki 4",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr20",
		URL:              "jpandkorea2025/P1217623.jpg",
		DateCreated:      time.Date(2025, 8, 8, 19, 33, 25, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr21",
		InfoCardTitle:    "NHK Nagasaki 3",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr19",
		URL:              "jpandkorea2025/P1217579.jpg",
		DateCreated:      time.Date(2025, 8, 8, 19, 32, 52, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr20",
		InfoCardTitle:    "NHK Nagasaki 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr18",
		URL:              "jpandkorea2025/P1217510.jpg",
		DateCreated:      time.Date(2025, 8, 8, 19, 30, 24, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr19",
		InfoCardTitle:    "NHK Nagasaki 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr17",
		URL:              "jpandkorea2025/P1217332.jpg",
		DateCreated:      time.Date(2025, 8, 8, 16, 3, 43, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr18",
		InfoCardTitle:    "長崎の猫",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr17-insert",
		URL:              "jpandkorea2025/P1217035.jpg",
		DateCreated:      time.Date(2025, 8, 8, 10, 43, 47, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr17",
		InfoCardTitle:    "長崎平和公園の折り鶴",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr16",
		URL:              "jpandkorea2025/P1216748.jpg",
		DateCreated:      time.Date(2025, 8, 7, 16, 16, 36, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr17-insert",
		InfoCardTitle:    "鷲尾愛宕神社",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr15",
		URL:              "jpandkorea2025/P1216424.jpg",
		DateCreated:      time.Date(2025, 8, 7, 11, 55, 10, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr16",
		InfoCardTitle:    "teamLab Forest",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr14",
		URL:              "jpandkorea2025/P1215876.jpg",
		DateCreated:      time.Date(2025, 8, 6, 16, 28, 52, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr15",
		InfoCardTitle:    "桜井二見ヶ浦 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr13",
		URL:              "jpandkorea2025/P1205679.jpg",
		DateCreated:      time.Date(2025, 8, 6, 16, 16, 30, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr14",
		InfoCardTitle:    "桜井二見ヶ浦 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr12",
		URL:              "jpandkorea2025/P1205415.jpg",
		DateCreated:      time.Date(2025, 8, 5, 20, 13, 37, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr13",
		InfoCardTitle:    "반포대교 달빛무지개분수",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr11",
		URL:              "jpandkorea2025/P1204894.jpg",
		DateCreated:      time.Date(2025, 8, 5, 13, 02, 21, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr12",
		InfoCardTitle:    "전쟁기념관",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr10",
		URL:              "jpandkorea2025/P1194724.jpg",
		DateCreated:      time.Date(2025, 8, 5, 10, 57, 43, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr11",
		InfoCardTitle:    "창덕궁 후원",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr9",
		URL:              "jpandkorea2025/P1194433.jpg",
		DateCreated:      time.Date(2025, 8, 4, 19, 49, 56, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr11",
		InfoCardTitle:    "낙산 성곽길",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr8",
		URL:              "jpandkorea2025/P1194095.jpg",
		DateCreated:      time.Date(2025, 8, 4, 16, 26, 06, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr9",
		InfoCardTitle:    "동대문디자인플라자 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr7",
		URL:              "jpandkorea2025/P1194112.jpg",
		DateCreated:      time.Date(2025, 8, 4, 16, 27, 51, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr8",
		InfoCardTitle:    "동대문디자인플라자 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr6",
		URL:              "jpandkorea2025/P1193834.jpg",
		DateCreated:      time.Date(2025, 8, 4, 10, 58, 27, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr7",
		InfoCardTitle:    "북촌한옥마을",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr5",
		URL:              "jpandkorea2025/P1193765.jpg",
		DateCreated:      time.Date(2025, 8, 4, 10, 05, 39, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr6",
		InfoCardTitle:    "Praying Mantis",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr4",
		URL:              "jpandkorea2025/P1183346.jpg",
		DateCreated:      time.Date(2025, 8, 4, 9, 23, 46, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr5",
		InfoCardTitle:    "경복궁 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr3",
		URL:              "jpandkorea2025/P1183651.jpg",
		DateCreated:      time.Date(2025, 8, 4, 9, 49, 46, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr4",
		InfoCardTitle:    "경복궁 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jpkr2",
		URL:              "jpandkorea2025/P1182876.jpg",
		DateCreated:      time.Date(2025, 8, 3, 19, 06, 55, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr4",
		InfoCardTitle:    "시청",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jpkr1",
		URL:              "jpandkorea2025/P1182840.jpg",
		DateCreated:      time.Date(2025, 8, 3, 18, 51, 4, 0, time.FixedZone("KST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr2",
		InfoCardTitle:    "명동",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp35",
		URL:              "japan2024/P1171844.jpg",
		DateCreated:      time.Date(2024, 7, 31, 19, 36, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jpkr1",
		InfoCardTitle:    "上空から見た夜の大阪",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp34",
		URL:              "japan2024/P1161321.jpg",
		DateCreated:      time.Date(2024, 7, 31, 18, 21, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp35",
		InfoCardTitle:    "空中庭園展望台",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp33",
		URL:              "japan2024/P1161208.jpg",
		DateCreated:      time.Date(2024, 7, 31, 13, 53, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp34",
		InfoCardTitle:    "クラゲ",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp32",
		URL:              "japan2024/P1160349.jpg",
		DateCreated:      time.Date(2024, 7, 31, 8, 59, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp33",
		InfoCardTitle:    "住𠮷大社",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp31",
		URL:              "japan2024/P1160147.jpg",
		DateCreated:      time.Date(2024, 7, 30, 21, 29, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp32",
		InfoCardTitle:    "道頓堀",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp30",
		URL:              "japan2024/P1159767.jpg",
		DateCreated:      time.Date(2024, 7, 29, 16, 12, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp32",
		InfoCardTitle:    "清水寺",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp29",
		URL:              "japan2024/P1149102.jpg",
		DateCreated:      time.Date(2024, 7, 29, 10, 23, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp30",
		InfoCardTitle:    "依水園",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp28",
		URL:              "japan2024/P1148810.jpg",
		DateCreated:      time.Date(2024, 7, 28, 7, 48, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp29",
		InfoCardTitle:    "嵐山竹林",
		InfoCardPosition: BLeftSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp27",
		URL:              "japan2024/P1148643.jpg",
		DateCreated:      time.Date(2024, 7, 27, 18, 56, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp28",
		InfoCardTitle:    "伏見稲荷大社 4",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp26",
		URL:              "japan2024/P1138611.jpg",
		DateCreated:      time.Date(2024, 7, 27, 18, 56, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp27",
		InfoCardTitle:    "伏見稲荷大社 3",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp25",
		URL:              "japan2024/P1138499.jpg",
		DateCreated:      time.Date(2024, 7, 27, 18, 44, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp26",
		InfoCardTitle:    "伏見稲荷大社 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp24",
		URL:              "japan2024/P1138361.jpg",
		DateCreated:      time.Date(2024, 7, 27, 18, 28, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp25",
		InfoCardTitle:    "伏見稲荷大社 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp23",
		URL:              "japan2024/P1138031.jpg",
		DateCreated:      time.Date(2024, 7, 26, 9, 57, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp24",
		InfoCardTitle:    "teamLab Planets 3",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp22",
		URL:              "japan2024/P1137830.jpg",
		DateCreated:      time.Date(2024, 7, 26, 9, 41, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp23",
		InfoCardTitle:    "teamLab Planets 2",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp21",
		URL:              "japan2024/P1137645.jpg",
		DateCreated:      time.Date(2024, 7, 26, 9, 24, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp23",
		InfoCardTitle:    "teamLab Planets 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp20",
		URL:              "japan2024/P1127586.jpg",
		DateCreated:      time.Date(2024, 7, 26, 9, 4, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp21",
		InfoCardTitle:    "テプコ豊洲ビル",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp19",
		URL:              "japan2024/P1127217.jpg",
		DateCreated:      time.Date(2024, 7, 25, 16, 24, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp20",
		InfoCardTitle:    "忠霊塔",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp18",
		URL:              "japan2024/P1127011.jpg",
		DateCreated:      time.Date(2024, 7, 25, 10, 49, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp20",
		InfoCardTitle:    "新宿御苑 2",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp17",
		URL:              "japan2024/P1126948.jpg",
		DateCreated:      time.Date(2024, 7, 25, 10, 41, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp18",
		InfoCardTitle:    "新宿御苑 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp16",
		URL:              "japan2024/P1126843.jpg",
		DateCreated:      time.Date(2024, 7, 24, 22, 32, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp18",
		InfoCardTitle:    "東京湾",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp15",
		URL:              "japan2024/P1126726.jpg",
		DateCreated:      time.Date(2024, 7, 24, 21, 59, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp16",
		InfoCardTitle:    "テレポートブリッジ",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp14",
		URL:              "japan2024/P1126568.jpg",
		DateCreated:      time.Date(2024, 7, 24, 10, 28, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp15",
		InfoCardTitle:    "明治神宮の飾り樽",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp13",
		URL:              "japan2024/P1126479.jpg",
		DateCreated:      time.Date(2024, 7, 23, 19, 47, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp15",
		InfoCardTitle:    "第58回葛飾納涼花火大会",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp12",
		URL:              "japan2024/P1115597.jpg",
		DateCreated:      time.Date(2024, 7, 22, 21, 40, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp13",
		InfoCardTitle:    "雨夜の新宿 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp11",
		URL:              "japan2024/P1115331.jpg",
		DateCreated:      time.Date(2024, 7, 22, 21, 9, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp12",
		InfoCardTitle:    "雨夜の新宿 1",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp10",
		URL:              "japan2024/P1115113.jpg",
		DateCreated:      time.Date(2024, 7, 22, 16, 0, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp11",
		InfoCardTitle:    "絵馬 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp09",
		URL:              "japan2024/P1115083.jpg",
		DateCreated:      time.Date(2024, 7, 22, 15, 55, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp10",
		InfoCardTitle:    "箱根神社",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp08",
		URL:              "japan2024/P1104829.jpg",
		DateCreated:      time.Date(2024, 7, 22, 14, 32, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp09",
		InfoCardTitle:    "箱根元宮 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp07",
		URL:              "japan2024/P1114971.jpg",
		DateCreated:      time.Date(2024, 7, 22, 14, 50, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp08",
		InfoCardTitle:    "箱根元宮 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp06",
		URL:              "japan2024/P1104535-cropped.webp",
		DateCreated:      time.Date(2024, 7, 22, 10, 41, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp08",
		InfoCardTitle:    "彫刻の森美術館",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 2996, Y: 2248},
		Video:            false,
	}, {
		ID:               "jp05",
		URL:              "japan2024/P1093894.jpg",
		DateCreated:      time.Date(2024, 7, 21, 9, 29, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp06",
		InfoCardTitle:    "浜離宮恩賜庭園",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp04",
		URL:              "japan2024/P1093765.jpg",
		DateCreated:      time.Date(2024, 7, 21, 9, 7, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp05",
		InfoCardTitle:    "汐留シティセンター",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp03",
		URL:              "japan2024/P1093541.jpg",
		DateCreated:      time.Date(2024, 7, 20, 10, 06, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp04",
		InfoCardTitle:    "東京スカイツリー",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	}, {
		ID:               "jp02",
		URL:              "japan2024/P1093431.jpg",
		DateCreated:      time.Date(2024, 7, 20, 9, 14, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp04",
		InfoCardTitle:    "絵馬 1",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "jp01",
		URL:              "japan2024/P1093354.jpg",
		DateCreated:      time.Date(2024, 7, 19, 20, 24, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp02",
		InfoCardTitle:    "夜の横浜 2",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3896, Y: 5192},
		Video:            false,
	},
	{
		ID:               "jp00",
		URL:              "japan2024/P1093211.jpg",
		DateCreated:      time.Date(2024, 7, 19, 20, 6, 0, 0, time.FixedZone("JST", 9)),
		PositionAbsolute: false,
		RelativeID:       "jp01",
		InfoCardTitle:    "夜の横浜 1",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	},
	{
		ID:               "d0",
		URL:              "starjump.webp",
		DateCreated:      time.Date(2024, 6, 26, 20, 44, 0, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "jp00",
		InfoCardTitle:    "Starjump",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5120, Y: 1440},
		Video:            false,
	}, {
		ID:               "c0",
		URL:              "tsutsuji.jpg",
		DateCreated:      time.Date(2024, 5, 14, 19, 33, 14, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "d0",
		InfoCardTitle:    "つつじ",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "b0",
		URL:              "java.webp",
		DateCreated:      time.Date(2024, 4, 18, 14, 44, 52, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "c0",
		InfoCardTitle:    "Java",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	}, {
		ID:               "a0",
		URL:              "march2024-sakura-P1082294.webp",
		DateCreated:      time.Date(2024, 3, 2, 21, 18, 20, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "b0",
		InfoCardTitle:    "夜桜 1",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	},
	{
		ID:               "a1",
		URL:              "march2024-sakura-P1082290.webp",
		DateCreated:      time.Date(2024, 3, 2, 21, 15, 49, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "a0",
		InfoCardTitle:    "夜桜 2",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	},
	{
		ID:               "a2",
		URL:              "march2024-sakura-P1082305.webp",
		DateCreated:      time.Date(2024, 3, 2, 21, 25, 54, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "a1",
		InfoCardTitle:    "夜桜 3",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	},

	{
		ID:               "a3",
		URL:              "march2024-sakura-P1082293.webp",
		DateCreated:      time.Date(2024, 3, 2, 21, 17, 33, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "a1",
		InfoCardTitle:    "夜桜 4",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 5192, Y: 3896},
		Video:            false,
	},
	{
		ID:               "0",
		URL:              "tokyowatercolor.webp",
		DateCreated:      time.Date(2024, 2, 21, 13, 07, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "a3",
		InfoCardTitle:    "東京・水彩画風",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 2560, Y: 1440},
		Video:            false,
	},
	{
		ID:               "1",
		URL:              "5yencoins.webp",
		DateCreated:      time.Date(2024, 02, 17, 14, 59, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "0",
		InfoCardTitle:    "For good luck.",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3360, Y: 1440},
		Video:            false,
	},
	{
		ID:               "2",
		URL:              "watercolordesign.mp4",
		DateCreated:      time.Date(2024, 02, 10, 20, 9, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "1",
		InfoCardTitle:    "Watercolor Design",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 1440, Y: 1440},
		Video:            true,
	},
	{
		ID:               "3",
		URL:              "GPTrainV8-yuv420p-compressed.mp4",
		DateCreated:      time.Date(2024, 01, 5, 16, 19, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "1",
		InfoCardTitle:    "Neon Subway",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 1920, Y: 1440},
		Video:            true,
	},
	{
		ID:               "4",
		URL:              "dreaming.jpg",
		DateCreated:      time.Date(2023, 12, 25, 18, 49, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "3",
		InfoCardTitle:    "Dreaming",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 4096, Y: 2160},
		Video:            false,
	},
	{
		ID:               "5",
		URL:              "krakowfoliage1.jpg",
		DateCreated:      time.Date(2023, 10, 28, 14, 38, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "4",
		InfoCardTitle:    "Foliage 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "6",
		URL:              "krakowfoliage2.jpg",
		DateCreated:      time.Date(2023, 10, 28, 13, 53, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "4",
		InfoCardTitle:    "Foliage 2",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "7",
		URL:              "saltmine1.jpg",
		DateCreated:      time.Date(2023, 10, 28, 9, 19, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "6",
		InfoCardTitle:    "Salt Mine 1",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "8",
		URL:              "saltmine2.jpg",
		DateCreated:      time.Date(2023, 10, 28, 9, 11, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "7",
		InfoCardTitle:    "Salt Mine 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "9",
		URL:              "saltmine3.jpg",
		DateCreated:      time.Date(2023, 10, 28, 9, 42, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "8",
		InfoCardTitle:    "Salt Mine 3",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "10",
		URL:              "saltmine4.jpg",
		DateCreated:      time.Date(2023, 10, 28, 8, 15, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "8",
		InfoCardTitle:    "Salt Mine 4",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "11",
		URL:              "ravens.jpg",
		DateCreated:      time.Date(2023, 10, 26, 17, 58, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "10",
		InfoCardTitle:    "Ravens",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "auschwitz",
		URL:              "auschwitz.jpg",
		DateCreated:      time.Date(2023, 10, 27, 12, 13, 33, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "10",
		InfoCardTitle:    "Auschwitz",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "12",
		URL:              "lilypads.jpg",
		DateCreated:      time.Date(2023, 8, 8, 10, 44, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "auschwitz",
		InfoCardTitle:    "Lilypads",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "13",
		URL:              "powerlines.jpg",
		DateCreated:      time.Date(2023, 7, 31, 17, 01, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "12",
		InfoCardTitle:    "Powerlines",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},

	{
		ID:               "14",
		URL:              "telecom.jpg",
		DateCreated:      time.Date(2023, 8, 5, 22, 16, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "13",
		InfoCardTitle:    "Telecom",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "15",
		URL:              "tsunamiwarning.jpg",
		DateCreated:      time.Date(2023, 8, 5, 22, 14, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "14",
		InfoCardTitle:    "Tsunami Warning",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "16",
		URL:              "khlong.jpg",
		DateCreated:      time.Date(2023, 8, 2, 10, 40, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "14",
		InfoCardTitle:    "Khlong 1",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "khlong2",
		URL:              "khlong2.jpg",
		DateCreated:      time.Date(2023, 8, 2, 10, 50, 13, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "16",
		InfoCardTitle:    "Khlong 2",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "lotus1",
		URL:              "lotus1.jpg",
		DateCreated:      time.Date(2023, 8, 2, 13, 16, 43, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "khlong2",
		InfoCardTitle:    "Lotus 1",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "lotus2",
		URL:              "lotus2.jpg",
		DateCreated:      time.Date(2023, 8, 2, 13, 17, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "lotus1",
		InfoCardTitle:    "Lotus 2",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 4940, Y: 3272},
		Video:            false,
	},
	{
		ID:               "17",
		URL:              "orchids.jpg",
		DateCreated:      time.Date(2023, 7, 31, 16, 29, 00, 0, time.FixedZone("ICT", 7)),
		PositionAbsolute: false,
		RelativeID:       "lotus1",
		InfoCardTitle:    "Orchids",
		InfoCardPosition: BTopSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3272, Y: 4940},
		Video:            false,
	},
	{
		ID:               "kanzansakura",
		URL:              "kanzansakura.webP",
		DateCreated:      time.Date(2023, 4, 1, 12, 27, 9, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "17",
		InfoCardTitle:    "Kanzan Sakura",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	},
	{
		ID:               "maple",
		URL:              "maple.jpg",
		DateCreated:      time.Date(2023, 5, 29, 12, 24, 34, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "kanzansakura",
		InfoCardTitle:    "Maple",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	},
	{
		ID:               "junethunderstorm",
		URL:              "junethunderstorm.jpg",
		DateCreated:      time.Date(2023, 6, 11, 20, 37, 50, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "maple",
		InfoCardTitle:    "Thunderstorm",
		InfoCardPosition: BTopSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	},
	{
		ID:               "18",
		URL:              "theblaze.png",
		DateCreated:      time.Date(2023, 5, 1, 15, 18, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "junethunderstorm",
		InfoCardTitle:    "The Blaze",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3334, Y: 1755},
		Video:            false,
	},
	{
		ID:               "19",
		URL:              "GreekArchitecture-HQ-Cam6-Signed.png",
		DateCreated:      time.Date(2023, 7, 10, 22, 02, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "18",
		InfoCardTitle:    "Greek Temple 1",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3840, Y: 2160},
		Video:            false,
	},

	{
		ID:               "19",
		URL:              "GreekArchitecture-HQ-Cam2-Signed.png",
		DateCreated:      time.Date(2023, 7, 10, 22, 02, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "18",
		InfoCardTitle:    "Greek Temple 2",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3840, Y: 2160},
		Video:            false,
	},
	{
		ID:               "20",
		URL:              "moebius-filter.png",
		DateCreated:      time.Date(2023, 8, 4, 2, 39, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "19",
		InfoCardTitle:    "Moebius Style",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	},
	{
		ID:               "21",
		URL:              "cyanotype.jpg",
		DateCreated:      time.Date(2023, 1, 18, 12, 40, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "19",
		InfoCardTitle:    "Cyanotype",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 1259, Y: 1259},
		Video:            false,
	},
	{
		ID:               "22",
		URL:              "apocalypticbuildings.png",
		DateCreated:      time.Date(2022, 11, 25, 23, 43, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "21",
		InfoCardTitle:    "Apocalyptic City",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 810, Y: 1080},
		Video:            false,
	},
	{
		ID:               "23",
		URL:              "railroad.webp",
		DateCreated:      time.Date(2022, 12, 19, 23, 54, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "21",
		InfoCardTitle:    "Railroad",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 2880, Y: 3600},
		Video:            false,
	},
	{
		ID:               "24",
		URL:              "wakes.jpg",
		DateCreated:      time.Date(2022, 11, 2, 17, 12, 00, 0, time.FixedZone("EET", 2)),
		PositionAbsolute: false,
		RelativeID:       "22",
		InfoCardTitle:    "Wakes",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5184, Y: 3888},
		Video:            false,
	},
	{
		ID:               "25",
		URL:              "olivetree.jpg",
		DateCreated:      time.Date(2022, 11, 2, 17, 15, 00, 0, time.FixedZone("EET", 2)),
		PositionAbsolute: false,
		RelativeID:       "23",
		InfoCardTitle:    "Olive",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	},
	{
		ID:               "26",
		URL:              "chuuhai.mp4",
		DateCreated:      time.Date(2022, 11, 6, 16, 25, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "25",
		InfoCardTitle:    "酎ハイ",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 836, Y: 1080},
		Video:            true,
	},
	{
		ID:               "27",
		URL:              "greeksunset.jpg",
		DateCreated:      time.Date(2022, 11, 1, 17, 44, 00, 0, time.FixedZone("EET", 2)),
		PositionAbsolute: false,
		RelativeID:       "26",
		InfoCardTitle:    "Greek Sunset",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 5184, Y: 3888},
		Video:            false,
	},
	{
		ID:               "28",
		URL:              "oceanlightbulb.webp",
		DateCreated:      time.Date(2022, 10, 22, 19, 27, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "26",
		InfoCardTitle:    "Lightbulb Benchy",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 2160, Y: 2160},
		Video:            false,
	},
	{
		ID:               "29",
		URL:              "wheatfield.jpg",
		DateCreated:      time.Date(2022, 10, 16, 21, 01, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "27",
		InfoCardTitle:    "小麦畑",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5120, Y: 1440},
		Video:            false,
	},
	{
		ID:               "30",
		URL:              "hummingbird.jpg",
		DateCreated:      time.Date(2022, 8, 2, 11, 28, 00, 0, time.FixedZone("AST", -4)),
		PositionAbsolute: false,
		RelativeID:       "28",
		InfoCardTitle:    "Hummingbird",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5184, Y: 3888},
		Video:            false,
	}, {
		ID:               "31",
		URL:              "longexposure.jpg",
		DateCreated:      time.Date(2022, 3, 19, 22, 16, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "30",
		InfoCardTitle:    "Godspeed",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5184, Y: 3888},
		Video:            false,
	}, {
		ID:               "32",
		URL:              "90sanimesubway.webp",
		DateCreated:      time.Date(2022, 7, 13, 22, 40, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "31",
		InfoCardTitle:    "Retro Subway",
		InfoCardPosition: BBottomSLeft,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3840, Y: 2880},
		Video:            false,
	}, {
		ID:               "plantsnearoldtraintracks",
		URL:              "plantsnearoldtraintracks.jpg",
		DateCreated:      time.Date(2021, 11, 24, 16, 35, 31, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "31",
		InfoCardTitle:    "Old Train Tracks",
		InfoCardPosition: BRightSBottom,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	}, {
		ID:               "oldtrainsignal",
		URL:              "oldtrainsignal.jpg",
		DateCreated:      time.Date(2021, 11, 24, 16, 25, 22, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "plantsnearoldtraintracks",
		InfoCardTitle:    "Old Train Signal",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	}, {
		ID:               "moonxpowerline",
		URL:              "moonxpowerline.webP",
		DateCreated:      time.Date(2021, 10, 31, 6, 18, 21, 0, time.FixedZone("MUT", 4)),
		PositionAbsolute: false,
		RelativeID:       "plantsnearoldtraintracks",
		InfoCardTitle:    "Moon & Powerlines",
		InfoCardPosition: BBottomSRight,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Full,
		Resolution:       utils.Coords{X: 3888, Y: 5184},
		Video:            false,
	}, {
		ID:               "33",
		URL:              "birds.webp",
		DateCreated:      time.Date(2021, 9, 6, 20, 11, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "moonxpowerline",
		InfoCardTitle:    "Twilight & Birds",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 5184, Y: 3888},
		Video:            false,
	}, {
		ID:               "34",
		URL:              "lightbulb.webp",
		DateCreated:      time.Date(2021, 11, 14, 12, 11, 00, 0, time.FixedZone("CET", 1)),
		PositionAbsolute: false,
		RelativeID:       "33",
		InfoCardTitle:    "Light Bulb",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         false,
		AddHeight:        true,
		Size:             Half,
		Resolution:       utils.Coords{X: 1920, Y: 1920},
		Video:            false,
	}, {
		ID:               "35",
		URL:              "apocalypticneon.jpg",
		DateCreated:      time.Date(2021, 5, 22, 15, 47, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "33",
		InfoCardTitle:    "Apocalyptic Neon",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 3840, Y: 2160},
		Video:            false,
	}, {
		ID:               "36",
		URL:              "blackhole.mp4",
		DateCreated:      time.Date(2021, 9, 26, 19, 35, 00, 0, time.FixedZone("CEST", 2)),
		PositionAbsolute: false,
		RelativeID:       "34",
		InfoCardTitle:    "Black Hole",
		InfoCardPosition: BRightSTop,
		Position:         utils.Coords{X: 0, Y: 0},
		AddWidth:         true,
		AddHeight:        false,
		Size:             Half,
		Resolution:       utils.Coords{X: 1080, Y: 1080},
		Video:            true,
	},
}
