use image::{ImageBuffer, RgbImage};
use rand::Rng;
use palette::{FromColor, Srgb, Hsl};
use std::time::{Instant};

struct Particle {
    velocity: [f64; 2],
    position: [f64; 2],
}

fn main() {
    let mut particles = Vec::new();
    let particle_count = 5000;
    let particle_mass = 1f64;
    let particle_radius = 2f64;
    let particle_collision_hardness = 20f64;
    let particle_collision_falloff = 2f64;
    let particle_collision_max_radius = 1.0f64;
    let gravity = 30f64;
    let drag_coefficient = 0.05f64;
    let border_bounciness_coefficient = 0.9f64;
    let generator_position = [200f64, 150f64];
    let generator_jitter = 10f64;
    let domain_size = [400f64, 300f64];

    let speed_hue_sensitivity = -2.0f64;
    let zero_speed_hue = 120.0f64;
    let image_size = [domain_size[0] as u32, domain_size[1] as u32];

    let time_step = 0.05f64;

    let iterations = 2000;

    let mut rng = rand::thread_rng();
    println!("Generating particles...");
    for _x in 0..particle_count {
        particles.push(Particle {
            velocity: [0.0, 0.0],
            position: [
                generator_position[0]
                    + (generator_jitter / 2f64 - generator_jitter * rng.gen_range(0.0..1.0)),
                generator_position[1]
                    + (generator_jitter / 2f64 - generator_jitter * rng.gen_range(0.0..1.0)),
            ],
        });
    }
    println!("Done.");
    println!("Let the simulation begin!");
    let simulation_start_time = Instant::now();
    for iteration_index in 0..iterations {
        // Calculate velocities
        for particle_index in 0..particle_count {
            // Inter particle collisions
            for particle_index_to_check in 0..particle_count {
                if particle_index_to_check != particle_index {
                    let x_diff = particles[particle_index].position[0]
                        - particles[particle_index_to_check].position[0];
                    let y_diff = particles[particle_index].position[1]
                        - particles[particle_index_to_check].position[1];
                    let total_dist = (x_diff.powf(2.0) + y_diff.powf(2.0)).sqrt();
                    if total_dist < particle_radius {
                        let max_diff = x_diff.abs().max(y_diff.abs());
                        let x_dir = x_diff / max_diff;
                        let y_dir = y_diff / max_diff;
                        let total_velocity = particle_collision_hardness
                            / (particle_collision_max_radius
                                + total_dist / particle_radius * particle_collision_falloff);
                        particles[particle_index].velocity[0] += x_dir * total_velocity * time_step;
                        particles[particle_index].velocity[1] += y_dir * total_velocity * time_step;
                    }
                }
            }
            // Gravity
            particles[particle_index].velocity[1] += gravity * time_step;
            //Drag
            let x_drag_force =
                drag_coefficient * ((particles[particle_index].velocity[0].powf(2.0)) / (2.0));
            let y_drag_force =
                drag_coefficient * ((particles[particle_index].velocity[1].powf(2.0)) / (2.0));
            let x_drag_acceleration = x_drag_force / particle_mass;
            let y_drag_acceleration = y_drag_force / particle_mass;
            let x_drag_velocity = x_drag_acceleration * time_step;
            let y_drag_velocity = y_drag_acceleration * time_step;
            if particles[particle_index].velocity[0] > 0.0 {
                particles[particle_index].velocity[0] -= x_drag_velocity;
            } else {
                particles[particle_index].velocity[0] += x_drag_velocity;
            }
            if particles[particle_index].velocity[1] > 0.0 {
                particles[particle_index].velocity[1] -= y_drag_velocity;
            } else {
                particles[particle_index].velocity[1] += y_drag_velocity;
            }
        }

        // Calculate positions from velocity.
        for particle_index in 0..particle_count {
            particles[particle_index].position[0] +=
                particles[particle_index].velocity[0] * time_step;
            particles[particle_index].position[1] +=
                particles[particle_index].velocity[1] * time_step;
            // Check for collisions with world border and rectify
            if particles[particle_index].position[0] < 0.0 {
                particles[particle_index].position[0] =
                -particles[particle_index].position[0] * border_bounciness_coefficient;
                particles[particle_index].velocity[0] =
                    -particles[particle_index].velocity[0] * border_bounciness_coefficient;
            }
            if particles[particle_index].position[0] > domain_size[0] {
                particles[particle_index].position[0] = domain_size[0]
                    + (domain_size[0] - (particles[particle_index].position[0]))
                        * border_bounciness_coefficient;
                particles[particle_index].velocity[0] =
                    -particles[particle_index].velocity[0] * border_bounciness_coefficient;
            }
            if particles[particle_index].position[1] < 0.0 {
                particles[particle_index].position[1] =
                -particles[particle_index].position[1] * border_bounciness_coefficient;
                particles[particle_index].velocity[1] =
                    -particles[particle_index].velocity[1] * border_bounciness_coefficient;
            }
            if particles[particle_index].position[1] > domain_size[1] {
                particles[particle_index].position[1] = domain_size[1]
                    + (domain_size[1] - (particles[particle_index].position[1]))
                        * border_bounciness_coefficient;
                particles[particle_index].velocity[1] =
                    -particles[particle_index].velocity[1] * border_bounciness_coefficient;
            }
        }
        // Render
        let mut img: RgbImage = ImageBuffer::new(image_size[0], image_size[1]);
        for particle_index in 0..particle_count {
            let rgb = Srgb::from_color(Hsl::new(zero_speed_hue+((particles[particle_index].velocity[0].powf(2.0)+particles[particle_index].velocity[1].powf(2.0)).sqrt()*speed_hue_sensitivity),1.0,0.5));
            img[(
                (((particles[particle_index].position[0]) - 1f64)
                    .round()
                    .min(domain_size[0] - 1f64)
                    .max(0.0) as u32),
                (((particles[particle_index].position[1] - 1.0) - 1f64)
                    .round()
                    .min(domain_size[1] - 1f64)
                    .max(0.0) as u32),
            )] = image::Rgb([(rgb.red*255f64.round()) as u8, (rgb.green*255f64.round()) as u8, (rgb.blue*255f64.round()) as u8]);
        }
        img.save(format!("./output/{:0>4}.png", iteration_index))
            .unwrap();
        // Display status to the user.
        let time_per_iteration =
            simulation_start_time.elapsed().as_millis() / (iteration_index + 1);
        println!(
            "Frame : {}/{}, Time elasped : {}s, Time per iteration: {}ms, Time left: {}s",
            iteration_index + 1,
            iterations,
            simulation_start_time.elapsed().as_secs(),
            time_per_iteration,
            (iterations - (iteration_index + 1)) * time_per_iteration / 1000
        );
    }
}
